const httpStatus = require('http-status');
const { Page } = require('../models');
const ApiError = require('../utils/ApiError');

/**
 * Create a page
 * @param {Object} body
 * @returns {Promise<User>}
 */
const createPage = async (body) => {
  const page = await Page.create(body);
  return page;
};

/**
 * Get All pages
 */
const getAllPages = async () => {
  return Page.find();
};

/**
 * Get user by id
 * @param {ObjectId} id
 * @returns {Promise<User>}
 */
const getPageById = async (id) => {
  return Page.findById(id);
};

/**
 * Get user by email
 * @param {string} email
 * @returns {Promise<User>}
 */
const getPageByTitle = async (email) => {
  return Page.findOne({ email });
};

/**
 * Update user by id
 * @param {ObjectId} userId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updatePageById = async (userId, updateBody) => {
  const page = await getPageById(userId);
  if (!page) {
    throw new ApiError(httpStatus.NOT_FOUND, 'Page not found');
  }
  Object.assign(page, updateBody);
  await page.save();
  return page;
};

/**
 * Delete user by id
 * @param {ObjectId} userId
 * @returns {Promise<User>}
 */
const deletePageById = async (userId) => {
  const page = await getPageById(userId);
  if (!page) {
    throw new ApiError(httpStatus.NOT_FOUND, 'Page not found');
  }
  await page.remove();
  return page;
};

module.exports = {
  createPage,
  getPageById,
  getAllPages,
  getPageByTitle,
  updatePageById,
  deletePageById,
};
