const httpStatus = require('http-status');
const otpGenerator = require('otp-generator');
const logger = require('../config/logger');
const { Meet, Attendees } = require('../models');
const ApiError = require('../utils/ApiError');

/**
 * Get active meeting by name
 * @param {*} name
 */
const getActiveMeetByName = async (name) => {
  return Meet.findOne({ meetingName: name, isActive: true });
};

/**
 * Get meeting by meetingCode
 * @param {ObjectId} meetingCode
 */
const getMeetingByCode = async (meetingCode) => {
  return Meet.findOne({ meetingCode });
};

/**
 * Get active meeting by meeting Id
 * @param {*} meetingCode
 */
const getActiveMeetingByCode = async (meetingCode) => {
  return Meet.findOne({ meetingCode, isActive: true });
};

/**
 * Get active meeting by meeting ID
 * @param {*} id
 * @returns
 */
const getActiveMeetByMeetingID = async (id) => {
  return Meet.findOne({ _id: id, isActive: true });
};

/**
 * Get meeting by query
 * @param {ObjectId} query
 */
const getMeetingByQuery = async (query) => {
  return Meet.findOne(query);
};

/**
 * Create a meet
 * @param {Object} reqBody
 * @returns {Promise<Meet>}
 */
const createNewMeeting = async (reqBody, server) => {
  const callDetails = reqBody;

  // Generate meeting ID
  callDetails.meetingCode = otpGenerator.generate(8, {
    upperCase: false,
    specialChars: false,
    digits: true,
    alphabets: false,
  });

  const exists = await getActiveMeetingByCode(callDetails.meetingCode);
  if (exists) {
    throw new ApiError(
      httpStatus.EXPECTATION_FAILED,
      'Meeting with same name already exists. Please try different meeting ID.'
    );
  }

  // Set server ID
  callDetails.serverId = server.id;
  callDetails.serverURL = server.serverURL;
  callDetails.isActive = true;

  // Write to DB
  const meet = await Meet.create(callDetails);
  await Attendees.create({
    meetingId: meet.id,
    attendees: [],
  });

  // Return ID
  return callDetails.meetingCode;
};

// CRON To terminate all idle sessions
const meetInactivityMonitor = () => {
  setInterval(async () => {
    const totalRec = await Meet.find({ isActive: true });

    totalRec.forEach(async (meet) => {
      const createdAtDate = new Date(meet.createdAt);
      const endedAtDate = new Date();
      // Calculate diff bet two dates
      const diffTime = Math.abs(endedAtDate - createdAtDate);
      // calculate diff in seconds
      const diffTimeInSecond = Math.floor(diffTime / 1000);

      await Meet.updateOne(
        { _id: meet._id },
        { isActive: false, endedAt: endedAtDate.toISOString(), duration: diffTimeInSecond }
      );
    });

    // Meet.updateMany({ isActive: true }, { isActive: false, endedAt: endedAtDate, duration: endedAtDate }, (err) => {
    //   if (err) {
    //     logger.error('Meet Inactivity Monitor Error:', err);
    //   }
    // });
  }, 60 * 1000);
};

// meetInactivityMonitor();

/**
 * Get active meeting by name
 * @param {*} name
 */
const getMeetByName = async (name) => {
  return Meet.findOne({ meetingName: name });
};

/**
 * Update user by id
 * @param {ObjectId} meetingCode
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updateMeeting = async (meetingCode, updateBody) => {
  const meet = await getActiveMeetingByCode(meetingCode);
  if (!meet) {
    throw new Error('Invalid Meeting Code');
  }

  Object.assign(meet, updateBody);
  await meet.save();

  return meet;
};

/**
 * Update attendees
 * @param {ObjectId} meetingCode
 * @param {Object} userDetails
 * @returns {Promise<User>}
 */
const insertAttendee = async (meetingCode, userDetails) => {
  const meet = await getActiveMeetingByCode(meetingCode);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Invalid Meeting Code');
  }

  Attendees.findOneAndUpdate({ meetingId: meet.id }, { $addToSet: { attendees: userDetails } }, function (error) {
    if (error) {
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Error occurred while joining meeting');
    }
  });
};

/**
 * Ping user by id
 * @param {ObjectId} meetingCode
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const pingAttendee = async (meetingCode) => {
  const meet = await getMeetingByCode(meetingCode);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Invalid Meeting Code');
  }

  Object.assign(meet, { isActive: true });
  await meet.save();

  // Attendees.find({ meetingId: meet.id, 'attendees.userId': userId }, async (err, res) => {
  //   if (err || res.length === 0) {
  //     return;
  //   }

  // });
};

/**
 * Update users in meeting
 * @param {ObjectId} meetingCode
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updateMeetingUser = async (meetingName, userName) => {
  Meet.findOne({ meetingName, isActive: true }, (e, meeting) => {
    const meet = meeting;
    if (e || !meet) {
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting does not exist');
    }

    if (!meet.isActive) {
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting has already ended');
    }

    meet
      .updateOne({ $addToSet: { attendees: userName } })
      .then(() => {
        meet.save();
        return meet;
      })
      .catch(() => {
        throw new ApiError(httpStatus.EXPECTATION_FAILED, e);
      });
  });
};

/**
 * Insert Chat in meeting
 * @param {ObjectId} meetingCode
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const insertChatMessage = async (meetingCode, chatObj) => {
  Meet.updateOne(
    { meetingCode, isActive: true },
    { $push: { conversation: { ...chatObj, time: new Date().toISOString() } } },
    (err) => {
      if (err) logger.error(err);
    }
  );
};

/**
 * Login with meetingid and meetingPassword
 * @param {string} meetingCode
 * @param {string} meetingPassword
 * @returns {Promise<User>}
 */
const validateMeetingPassword = async (meetingCode, meetingPassword) => {
  const meet = await getActiveMeetingByCode(meetingCode);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Invalid Meeting Code');
  }

  if (meetingPassword !== meet.meetingPassword) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Incorrect Password');
  }

  return meet;
};

// meetInactivityMonitor();
module.exports = {
  createNewMeeting,
  updateMeeting,
  validateMeetingPassword,
  getMeetingByCode,
  updateMeetingUser,
  getMeetByName,
  getActiveMeetingByCode,
  insertAttendee,
  pingAttendee,
  getActiveMeetByName,
  insertChatMessage,
  getActiveMeetByMeetingID,
  getMeetingByQuery,
};
