const AWS = require('aws-sdk');
const { Media } = require('../models');
const config = require('../config/config');

const s3 = new AWS.S3({
  accessKeyId: config.s3.id,
  secretAccessKey: config.s3.secret,
});

/**
 * Create a media
 * @param {Object} body
 * @returns {Promise<User>}
 */
const insertNewMedia = async (body) => {
  const media = await Media.create(body);
  return media;
};

/**
 * Get All pages
 */
const getAllMedia = async () => {
  return Media.find();
};

/**
 * Get user by id
 * @param {ObjectId} id
 * @returns {Promise<User>}
 */
const getMediaById = async (id) => {
  return Media.findById(id);
};

/**
 * Get user by hostEmail
 * @param {string} hostEmail
 * @returns {Promise<User>}
 */
const getMediaByUserId = async (hostEmail) => {
  return Media.find({ hostEmail });
};

/**
 * Get media by query
 * @param {string} query
 * @returns {Promise<User>}
 */
const getMediaByQuery = async (query) => {
  return Media.find(query);
};

/**
 * Get single media by query
 * @param {string} query
 * @returns {Promise<User>}
 */
const getSingleMediaByQuery = async (query) => {
  return Media.findOne(query);
};

/**
 * Update user by id
 * @param {ObjectId} mediaId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updateMediaById = async (mediaId, updateBody) => {
  const media = await getMediaById(mediaId);
  if (!media) {
    throw new Error('Media not found');
  }
  Object.assign(media, updateBody);
  await media.save();
  return media;
};

/**
 * Delete user by id
 * @param {ObjectId} mediaId
 * @returns {Promise<User>}
 */
const deleteMediaById = async (mediaId) => {
  const media = await getMediaById(mediaId);
  if (!media) {
    throw new Error('Media not found');
  }
  await media.remove();
  return media;
};

/**
 * Delete recording from S3
 */
const deleteS3Recording = async (fileName) => {
  return new Promise((resolve, reject) => {
    s3.deleteObject(
      {
        Bucket: config.s3.bucketName,
        Key: fileName,
      },
      (err, data) => {
        if (err) return reject(new Error(err));
        return resolve(data);
      }
    );
  });
};

module.exports = {
  insertNewMedia,
  getMediaById,
  getAllMedia,
  getMediaByUserId,
  updateMediaById,
  deleteMediaById,
  getMediaByQuery,
  getSingleMediaByQuery,
  deleteS3Recording,
};
