const httpStatus = require('http-status');
const { v4: uuidv4 } = require('uuid');
const { License } = require('../models');
const ApiError = require('../utils/ApiError');

/**
 * Create a license
 * @param {Object} licenseBody
 * @returns {Promise<License>}
 */
const createLicense = async (licenseBody) => {
  const licenseKey = uuidv4();
  if (await License.isLicenseTaken(licenseKey)) {
    throw new Error('License key duplication. Please try again.');
  }

  /* eslint-disable */
  const license = await License.create({
    licenseKey,
    email: licenseBody.email,
    userLimit: licenseBody.userLimit,
    expiryDays: licenseBody.expiryDays,
    renewedAt: new Date().toISOString(),
    expiryDate:
      licenseBody.expiryDays === 'never'
        ? 'never'
        : new Date(new Date().getTime() + (parseInt(licenseBody.expiryDays) * 24 * 60 * 60 * 1000)).toISOString()
  });

  return { status: 'created' };
};

/**
 * Checks license / API Key validity for meeting API's
 * @param {*} licenseKey 
 * @returns 
 */
const checkLicenseValidity = async (licenseKey) => {
  const license = await License.findOne({ licenseKey: licenseKey });
  if (!license || license.status === 'inactive') {
    throw new Error('Invalid API key');
  }

  if (license.status === 'expired' || new Date(license.expiryDate) < new Date()) {
    throw new Error('API key has expired')
  }

  if (license.status === 'blocked') {
    throw new Error('API key has been blocked')
  }

  return license;
};

const queryLicense = async (query) => {
  const licenses = await License.find(query);
  return licenses;
};

/**
 * Get license by id
 * @param {ObjectId} id
 * @returns {Promise<License>}
 */
const getLicenseById = async (id) => {
  return License.findById(id);
};

/**
 * Get license by license Key
 * @param {ObjectId} key
 * @returns {Promise<License>}
 */
const getLicenseByKey = async (key) => {
  return License.queryLicense({ licenseKey: key });
};

/**
 * Get user by email
 * @param {string} email
 * @returns {Promise<License>}
 */
const getLicenseByEmail = async (email) => {
  return License.findOne({ email });
};

/**
 * Update license by id
 * @param {ObjectId} licenseId
 * @param {Object} updateBody
 * @returns {Promise<License>}
 */
const updateLicenseById = async (licenseId, updateBody) => {
  const license = await getLicenseById(licenseId);
  if (!license) {
    throw new ApiError(httpStatus.NOT_FOUND, 'License not found');
  }

  Object.assign(license, updateBody);
  await license.save();
  return license;
};

/**
 * Renew license by id
 * @param {ObjectId} licenseId
 * @param {Object} updateBody
 * @returns {Promise<License>}
 */
const renewLicenseById = async (licenseId, updateBody) => {
  const license = await getLicenseById(licenseId);
  if (!license) {
    throw new ApiError(httpStatus.NOT_FOUND, 'License not found');
  }

  Object.assign(license, {
    email: updateBody.email,
    userLimit: updateBody.userLimit,
    expiryDays: updateBody.expiryDays,
    status: 'active',
    renewedAt: new Date().toISOString(),
    expiryDate:
      updateBody.expiryDays === 'never'
        ? 'never'
        : new Date(new Date().getTime() + (parseInt(updateBody.expiryDays) * 24 * 60 * 60 * 1000)).toISOString()
  });

  await license.save();
  return license;
};


/**
 * Update user password by id
 * @param {ObjectId} userId
 * @param {Object} requestBody
 * @returns {Promise<License>}
 */
const changeUserPassword = async (userId, requestBody) => {
  const user = await getLicenseById(userId);
  if (!user || !(await user.isPasswordMatch(requestBody.currentPassword))) {
    throw new ApiError(httpStatus.UNAUTHORIZED, 'Incorrect current password');
  }

  const result = await updateLicenseById(userId, { password: requestBody.newPassword });
  return result;
};

/**
 * Delete user by id
 * @param {ObjectId} userId
 * @returns {Promise<License>}
 */
const deleteUserById = async (userId) => {
  const user = await getLicenseById(userId);
  if (!user) {
    throw new ApiError(httpStatus.NOT_FOUND, 'License not found');
  }
  await user.remove();
  return user;
};

// CRON To check licenses
const licenseMonitor = () => {
  setInterval(async () => {
    const licenses = await License.find({ status: 'active' })
    if (licenses.length > 0) {
      licenses.forEach(async license => {
        if (new Date() > new Date(license.expiryDate)) {
          await License.updateOne({ _id: license._id }, { status: 'expired' })
        }
      })
    }
  }, 5 * 1000);
};

licenseMonitor()

module.exports = {
  createLicense,
  updateLicenseById,
  renewLicenseById,
  queryLicense,
  getLicenseById,
  getLicenseByEmail,
  getLicenseByKey,
  deleteUserById,
  changeUserPassword,
  checkLicenseValidity
};
