const httpStatus = require('http-status');
const otpGenerator = require('otp-generator');
const { Calls } = require('../models');
const ApiError = require('../utils/ApiError');
const meetService = require('./meet.service');

/**
 * Create a meet
 * @param {Object} requestBody
 * @returns {Promise<Calls>}
 */
const createMeeting = async (requestBody, server) => {
  const meetingDetails = requestBody;

  // Generate meeting ID
  meetingDetails.meetingId = otpGenerator.generate(8, {
    upperCase: false,
    specialChars: false,
    digits: true,
    alphabets: false,
  });

  meetingDetails.isPasswordEnabled = String(meetingDetails.meetingPassword).trim().length > 0;

  // Set server ID
  meetingDetails.serverId = server.id;
  meetingDetails.serverURL = server.serverURL;

  // Check if meeting with same ID exists
  const meet = await meetService.getActiveMeetByName(meetingDetails.meetingName);
  if (meet) {
    if (meetingDetails.hostId === meet.hostId) {
      return meet.meetingId;
    }
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting with same name already exists');
  }

  // Write to DB
  await Calls.create(meetingDetails);

  // Return ID
  return meetingDetails.meetingId;
};

/**
 * Update user by id
 * @param {ObjectId} meetingId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updateMeeting = async (meetingId, updateBody) => {
  Calls.findOne({ meetingId }, async (e, meeting) => {
    const meet = meeting;
    if (e || !meet) {
      throw new ApiError(httpStatus.NOT_FOUND, 'Meeting does not exist');
    }

    Object.assign(meet, updateBody);
    await meet.save();
    return meet;
  });
};

/**
 * Update user by id
 * @param {ObjectId} meetingId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const insertAttendee = async (meetingId, userDetails) => {
  Calls.findOneAndUpdate({ meetingId }, { $addToSet: { attendees: userDetails } }, function (error) {
    if (error) {
      throw new ApiError(httpStatus.NOT_FOUND, 'Error updating attendee');
    }
  });
};

/**
 * Ping user by id
 * @param {ObjectId} meetingId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const pingAttendee = async (meetingId, userId) => {
  Calls.findOneAndUpdate(
    { meetingId, 'attendees.id': userId },
    { $set: { 'attendees.$.isActive': true } },
    function (error) {
      if (error) {
        throw new ApiError(httpStatus.NOT_FOUND, 'Error updating attendee');
      }
    }
  );
};

/**
 * Update users in meeting
 * @param {ObjectId} meetingId
 * @param {Object} updateBody
 * @returns {Promise<User>}
 */
const updateMeetingUser = async (meetingName, userName) => {
  Calls.findOne({ meetingName, isActive: true }, (e, meeting) => {
    const meet = meeting;
    if (e || !meet) {
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting does not exist');
    }

    if (!meet.isActive) {
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting has already ended');
    }

    meet
      .updateOne({ $addToSet: { attendees: userName } })
      .then(() => {
        meet.save();
        return meet;
      })
      .catch(() => {
        throw new ApiError(httpStatus.EXPECTATION_FAILED, e);
      });
  });
};

/**
 * Login with meetingid and meetingPassword
 * @param {string} meetingId
 * @param {string} meetingPassword
 * @returns {Promise<User>}
 */
const validateMeetingPassword = async (meetingId, meetingPassword) => {
  const meet = await meetService.getActiveMeetById(meetingId);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting has ended. Cannot proceed.');
  }

  if (meetingPassword !== meet.meetingPassword) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Incorrect Password');
  }

  return true;
};

module.exports = {
  createMeeting,
  updateMeeting,
  validateMeetingPassword,
  updateMeetingUser,
  insertAttendee,
  pingAttendee,
};
