const express = require('express');
const validate = require('../../middlewares/validate');
const licenseValidation = require('../../validations/license.validation');
const licenseController = require('../../controllers/license.controller');

const router = express.Router();

router.get('/get', licenseController.getAllLicenses);
router.post('/create', validate(licenseValidation.create), licenseController.createLicense);
router.post('/update', validate(licenseValidation.edit), licenseController.updateLicense);

module.exports = router;

/**
 * @swagger
 * tags:
 *   name: Licenses
 *   description: License Management
 */

/**
 * @swagger
 * path:
 *  /license/create:
 *    post:
 *      summary: Create a license
 *      description: Only admins can create new licenses
 *      tags: [Licenses]
 *      security:
 *        - bearerAuth: []
 *      requestBody:
 *        required: true
 *        content:
 *          application/json:
 *            schema:
 *              type: object
 *              required:
 *                - email
 *                - userLimit
 *                - expiryDays
 *              properties:
 *                email:
 *                  type: string
 *                userLimit:
 *                  type: number
 *                  format: integer
 *                  minLength: 1
 *                  description: At least one number
 *                expiryDays:
 *                  type: string
 *                  format: string
 *                  minLength: 1
 *                  description: At least one number
 *              example:
 *                email: imtiyazshaikh5786@gmail.com
 *                userLimit: 50
 *                expiryDays: "60"
 *      responses:
 *        "200":
 *          description: License Created
 */

/**
 * @swagger
 * path:
 *  /license/update:
 *    post:
 *      summary: Update a license
 *      description: Only admins can update new licenses
 *      tags: [Licenses]
 *      security:
 *        - bearerAuth: []
 *      requestBody:
 *        required: true
 *        content:
 *          application/json:
 *            schema:
 *              type: object
 *              required:
 *                - licenseId
 *                - updateBody
 *              properties:
 *                licenseId:
 *                  type: string
 *                updateBody:
 *                  type: object
 *                  description: JSON Object to update
 *              example:
 *                licenseId: "6060c839639d95485080fe44"
 *                updateBody: {
 *                  status: "inactive"
 *              }
 *      responses:
 *        "200":
 *          description: License Updated
 */

/**
 * @swagger
 * path:
 *  /license/get:
 *    get:
 *      summary: Get all licenses
 *      description: Only admins can fetch all licenses
 *      tags: [Licenses]
 *      security:
 *        - bearerAuth: []
 *      requestBody:
 *        required: false
 *      responses:
 *        "200":
 *          description: License Response
 */
