const httpStatus = require('http-status');
const ApiError = require('../utils/ApiError');
const catchAsync = require('../utils/catchAsync');
const { userService } = require('../services');

const createUser = catchAsync(async (req, res) => {
  const user = await userService.createUser(req.body);
  res.status(httpStatus.CREATED).send(user);
});

const getUsers = catchAsync(async (req, res) => {
  const result = await userService.getAllUsers();
  res.send(result);
});

const getUser = catchAsync(async (req, res) => {
  const user = await userService.getUserById(req.params.userId);
  if (!user) {
    throw new ApiError(httpStatus.NOT_FOUND, 'User not found');
  }
  res.send(user);
});

const updateUser = catchAsync(async (req, res) => {
  const user = await userService.updateUserById(req.body.userId, req.body);
  res.send(user);
});

const deleteUser = catchAsync(async (req, res) => {
  await userService.deleteUserById(req.params.userId);
  res.status(httpStatus.NO_CONTENT).send();
});

const updateUserProfile = catchAsync(async (req, res) => {
  const user = await userService.updateUserById(req.body.userId, req.body);
  res.status(httpStatus.OK).send(user);
});

const updateUserProfileImage = catchAsync(async (req, res) => {
  if (!req.files || Object.keys(req.files).length === 0) {
    return res.status(400).send('No files were uploaded.');
  }

  // The name of the input field (i.e. "sampleFile") is used to retrieve the uploaded file
  let { profileImage } = req.files;
  profileImage = req.files.profileImage;
  const path = `/profile-images/user-${req.body.userId}.jpg`;

  // Use the mv() method to place the file somewhere on your server
  profileImage.mv(`./src/uploads${path}`, function (err) {
    if (err) {
      return res.status(500).send(err);
    }
  });
  req.body.profileImage = path;

  const user = await userService.updateUserById(req.body.userId, req.body);
  res.status(httpStatus.OK).send(user);
});

const changeUserPassword = catchAsync(async (req, res) => {
  const user = await userService.changeUserPassword(req.body.userId, req.body);
  res.status(httpStatus.OK).send(user);
});

module.exports = {
  createUser,
  getUsers,
  getUser,
  updateUser,
  deleteUser,
  updateUserProfile,
  updateUserProfileImage,
  changeUserPassword,
};
