const httpStatus = require('http-status');
const catchAsync = require('../utils/catchAsync');
const { meetService, serverService, licenseService } = require('../services');
const ApiError = require('../utils/ApiError');

// Create new meeting
const createNewMeeting = catchAsync(async (req, res) => {
  try {
    const { apiKey } = req.body;

    await licenseService.checkLicenseValidity(apiKey);
    const server = await serverService.getAvailableServer();
    if (!server || !server.serverURL) throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Server not available');

    const meetingCode = await meetService.createNewMeeting(req.body, server);
    res.status(httpStatus.OK).send({ meetingCode, serverUrl: server.serverURL });
  } catch (e) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, e);
  }
});

// Get meeting by meeting code
const getMeetingByCode = catchAsync(async (req, res) => {
  const { meetingCode, userId } = req.body;
  const meet = await meetService.getMeetingByCode(meetingCode);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting does not exist');
  }

  return res.status(httpStatus.OK).send({
    meetingName: meet.meetingName,
    isActive: meet.isActive,
    meetingType: meet.sessionType,
    isPasswordEnabled: meet.isPasswordEnabled,
    isSessionHost: userId === meet.hostId,
    serverUrl: meet.serverURL,
  });
});

// Validate meeting and return server object
const validateMeeting = catchAsync(async (req, res) => {
  const { meetingCode, meetingPassword, userId } = req.body;
  const meeting = await meetService.validateMeetingPassword(meetingCode, meetingPassword);
  return res.status(httpStatus.OK).send({
    isSessionHost: userId === meeting.hostId,
    serverURL: meeting.serverURL,
  });
});

// Join meeting
const joinMeeting = catchAsync(async (req, res) => {
  const { meetingCode, userId, userName } = req.body;

  // Update meeting status
  await meetService.updateMeeting(meetingCode, { isActive: true });

  // Update attendee in the document
  await meetService.insertAttendee(meetingCode, {
    userId,
    userName,
  });

  return res.status(httpStatus.OK).send('OK');
});

// Ping for presence
const pingAttendee = catchAsync(async (req, res) => {
  const { meetingCode, userId } = req.body;
  await meetService.pingAttendee(meetingCode, userId);
  res.status(httpStatus.OK).send('OK');
});

// Reopen meeting room
const reopenMeeting = catchAsync(async (req, res) => {
  const { meetingCode, userId } = req.body;
  const meeting = await meetService.getMeetingByCode(meetingCode);
  if (!meeting) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Invalid meeting code');
  }

  if (meeting.hostId !== userId) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Unauthorized permission');
  }

  Object.assign(meeting, { isActive: true });
  await meeting.save();
  res.status(httpStatus.OK).send();
});

const updateMeeting = catchAsync(async (req, res) => {
  await meetService.updateMeeting(req.body.meetingCode, req.body);
  res.status(httpStatus.CREATED).send();
});

const updateMeetingUser = catchAsync(async (req, res) => {
  const { meetingName, userName } = req.body;
  await meetService.updateMeetingUser(meetingName, userName);
  res.send(true);
});

const updateMeetingChat = catchAsync(async (req, res) => {
  const { meetingCode, chatObj } = req.body;
  await meetService.insertChatMessage(meetingCode, chatObj);
  res.send(true);
});

const getChatByMeetingId = catchAsync(async (req, res) => {
  const meet = await meetService.getActiveMeetById(req.body.meetingCode);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting does not exist');
  }

  res.send(meet.conversation);
});

const getActiveMeetingByName = catchAsync(async (req, res) => {
  const meet = await meetService.getActiveMeetByName(req.body.meetingName);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting does not exist');
  }

  res.send({
    hostId: meet.hostId,
    hostName: meet.hostName,
    meetingCode: meet.meetingCode,
    meetingName: meet.meetingName,
    serverURL: meet.serverURL,
    isPasswordEnabled: meet.isPasswordEnabled,
    conversation: meet.conversation,
  });
});


// Create new meeting
const createNewMeetingForOthers = catchAsync(async (req, res) => {
  try {
    const { apiKey } = req.body;

    await licenseService.checkLicenseValidity(apiKey);
    const server = await serverService.getAvailableServer();
    if (!server || !server.serverURL) throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Server not available');

    const meetingCode = await meetService.createNewMeeting(req.body, server);
    res.status(httpStatus.OK).send({ meetingCode, serverUrl: server.serverURL });
  } catch (e) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, e);
  }
});

module.exports = {
  createNewMeeting,
  getMeetingByCode,
  validateMeeting,
  joinMeeting,
  updateMeeting,
  updateMeetingUser,
  updateMeetingChat,
  getChatByMeetingId,
  getActiveMeetingByName,
  pingAttendee,
  reopenMeeting,
  createNewMeetingForOthers,
};
