const httpStatus = require('http-status');
const path = require('path');
const ApiError = require('../utils/ApiError');
const catchAsync = require('../utils/catchAsync');
const uploader = require('../utils/FileUpload');
const logger = require('../config/logger');
const { mediaService, meetService } = require('../services');
const utils = require('../utils/utils');

const uploadScreenshot = catchAsync(async (req, res) => {
  uploader.screenShotUploader(req, res, async function (err) {
    if (err) {
      return new ApiError(httpStatus.EXPECTATION_FAILED, 'Failed to upload image');
    }

    const { userId, meetingId, mediaType, meetingName } = req.body;
    try {
      await mediaService.insertNewMedia({
        userId,
        meetingId,
        mediaType,
        meetingName,
        mediaPath: `/content/uploads/screenshots/${path.basename(req.file.path)}`,
      });

      res.status(httpStatus.OK).send(true);
    } catch (e) {
      logger.error(e);
      return new ApiError(httpStatus.EXPECTATION_FAILED, 'Failed to upload image');
    }
  });
});

const createNewRecording = catchAsync(async (req, res) => {
  const { validationCode, hostEmail, meetingCode, mediaUrl, alternateMediaUrl, fileName, fileSize } = req.body;

  if (validationCode !== 'mediaserverrequestforrecordingentry') {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Invalid Request');
  }

  const meeting = await meetService.getMeetingByQuery({ meetingCode, hostEmail });
  if (!meeting) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting not found');
  }

  await mediaService.insertNewMedia({
    hostName: meeting.hostName,
    hostEmail,
    meetingCode: meeting.meetingCode,
    meetingId: meeting.id,
    mediaType: 'recording',
    mediaUrl,
    fileName,
    fileSize,
    alternateMediaUrl,
  });

  res.status(httpStatus.OK).send(true);
});

const getUserScreenshots = catchAsync(async (req, res) => {
  const { hostEmail } = req.body;
  const mediaFiles = await mediaService.getMediaByQuery({ hostEmail, mediaType: 'screenshot' });
  res.status(httpStatus.OK).send(mediaFiles);
});

const getUserRecordings = catchAsync(async (req, res) => {
  const { hostEmail } = req.body;
  const mediaFiles = await mediaService.getMediaByQuery({ hostEmail, mediaType: 'recording' });
  res.status(httpStatus.OK).send(mediaFiles);
});

const getAllRecordings = catchAsync(async (req, res) => {
  const mediaFiles = await mediaService.getMediaByQuery({ mediaType: 'recording' });
  res.status(httpStatus.OK).send(mediaFiles);
});

const getAllScreenshots = catchAsync(async (req, res) => {
  const mediaFiles = await mediaService.getMediaByQuery({ mediaType: 'screenshot' });
  res.status(httpStatus.OK).send(mediaFiles);
});

const deleteMedia = catchAsync(async (req, res) => {
  const { hostEmail, meetingCode, mediaUrl, type } = req.body;
  const media = await mediaService.getSingleMediaByQuery({ mediaType: type, hostEmail, meetingCode, mediaUrl });
  if (!media) throw new ApiError('Media details not found');

  const fileName = utils.getFileNameFromURL(media.mediaUrl);
  await mediaService.deleteS3Recording(fileName);
  await mediaService.deleteMediaById(media._id);
  res.status(httpStatus.OK).send(true);
});

module.exports = {
  createNewRecording,
  uploadScreenshot,
  getUserScreenshots,
  getAllRecordings,
  getUserRecordings,
  getAllScreenshots,
  deleteMedia,
};
