const httpStatus = require('http-status');
const catchAsync = require('../utils/catchAsync');
const { apiService, meetService, serverService } = require('../services');
const ApiError = require('../utils/ApiError');

const createMeeting = catchAsync(async (req, res) => {
  try {
    const server = await serverService.getAvailableServer();
    if (!server || !server.id || !server.serverURL)
      throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Server not available');

    const meetingId = await apiService.createMeeting(req.body, server);
    res.status(httpStatus.OK).send({ meetingId, serverURL: server.serverURL });
  } catch (e) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, e);
  }
});

const updateMeeting = catchAsync(async (req, res) => {
  await meetService.updateMeeting(req.body.meetingId, req.body);
  res.status(httpStatus.Ok).send();
});

const insertAttendee = catchAsync(async (req, res) => {
  await meetService.insertAttendee(req.body.meetingId, req.body.user);
  res.status(httpStatus.OK).send();
});

const pingAttendee = catchAsync(async (req, res) => {
  await meetService.pingAttendee(req.body.meetingId, req.body.userId);
  res.status(httpStatus.OK).send();
});

const getActiveMeetingIdByMeetingName = catchAsync(async (req, res) => {
  const meet = await meetService.getActiveMeetByName(req.body.meetingName);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting is not active');
  }

  res.send({
    meetingId: meet.meetingId,
  });
});

const getMeetingByID = catchAsync(async (req, res) => {
  const meet = await meetService.getMeetById(req.body.meetingId);
  if (!meet) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting is not active');
  }

  if (!meet.isActive) {
    throw new ApiError(httpStatus.EXPECTATION_FAILED, 'Meeting has ended');
  }

  res.send({
    id: meet.id,
    meetingName: meet.meetingName,
    isActive: meet.isActive,
    isPasswordEnabled: meet.isPasswordEnabled,
    serverURL: meet.serverURL,
    isSessionHost: req.body.userId === meet.hostId,
  });
});

const validateMeeting = catchAsync(async (req, res) => {
  const { meetingId, password } = req.body;
  await meetService.validateMeetingPassword(meetingId, password);
  res.send(true);
});

module.exports = {
  createMeeting,
  getMeetingByID,
  updateMeeting,
  insertAttendee,
  pingAttendee,
  validateMeeting,
  getActiveMeetingIdByMeetingName,
};
